using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Text;

/// <summary>
/// NovaGodina - Finding e-similar spanning tree
/// 
/// Problem: Given n points in unit square [0,1]×[0,1], compute MST using Manhattan distance.
/// Select subset Q ? P with |Q| = 18/e² and build spanning tree T over Q
/// such that T and MST are e-similar.
/// 
/// e-similarity definition:
/// - For every point on MST (including edge segments), exists point on T within distance e
/// - For every point on T (including edge segments), exists point on MST within distance e
/// 
/// Key observations:
/// 1. If we sample MST vertices at intervals = e/2, every MST point is within e/2 of a sample
/// 2. T edges connect samples. Direct Manhattan distance = MST path length
/// 3. So T edge length = e (since samples are = e/2 apart along MST)
/// 4. Points on T edges of length = e are within e/2 of endpoints (which are on MST)
/// 
/// Algorithm:
/// 1. Special case: if e = 2 (unit square diagonal), return any single point
/// 2. Compute MST in O(n²) using Prim's algorithm
/// 3. Root MST at vertex 0 and do DFS
/// 4. Select: root, all leaves, all branching points, and vertices at distance intervals = e/2
/// 5. Connect selected vertices to form spanning tree T
/// 6. Verify |Q| = 18/e² (should always pass with our algorithm)
/// 
/// Time Complexity: O(n²) for MST + O(n) for DFS = O(n²)
/// Space Complexity: O(n)
/// </summary>
class NovaGodina
{
    static int n;
    static double epsilon;
    static double[] x, y;
    static List<int>[] mstAdj;
    
    // Fast I/O
    static StreamReader reader;
    static StringBuilder output;
    
    static double ManhattanDist(int i, int j)
    {
        return Math.Abs(x[i] - x[j]) + Math.Abs(y[i] - y[j]);
    }
    
    /// <summary>
    /// Prim's MST algorithm - O(n²) 
    /// For complete graphs, O(n²) is optimal since we must examine all n² edges.
    /// Simple array-based implementation is actually faster than priority queue for dense graphs.
    /// </summary>
    static void ComputeMST()
    {
        mstAdj = new List<int>[n];
        for (int i = 0; i < n; i++)
            mstAdj[i] = new List<int>();
        
        bool[] inMST = new bool[n];
        double[] minDist = new double[n];
        int[] parent = new int[n];
        
        for (int i = 0; i < n; i++)
        {
            minDist[i] = double.MaxValue;
            parent[i] = -1;
        }
        minDist[0] = 0;
        
        for (int iter = 0; iter < n; iter++)
        {
            // Find vertex with minimum distance not yet in MST
            int u = -1;
            double bestDist = double.MaxValue;
            
            for (int i = 0; i < n; i++)
            {
                if (!inMST[i] && minDist[i] < bestDist)
                {
                    bestDist = minDist[i];
                    u = i;
                }
            }
            
            if (u == -1) break;
            inMST[u] = true;
            
            // Add MST edge to adjacency list
            if (parent[u] != -1)
            {
                mstAdj[u].Add(parent[u]);
                mstAdj[parent[u]].Add(u);
            }
            
            // Relax edges from u
            for (int v = 0; v < n; v++)
            {
                if (!inMST[v])
                {
                    double dist = ManhattanDist(u, v);
                    if (dist < minDist[v])
                    {
                        minDist[v] = dist;
                        parent[v] = u;
                    }
                }
            }
        }
    }
    
    /// <summary>
    /// Fast line reading and parsing
    /// </summary>
    static double[] ReadDoubles()
    {
        string line = reader.ReadLine();
        string[] parts = line.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
        double[] result = new double[parts.Length];
        for (int i = 0; i < parts.Length; i++)
        {
            result[i] = double.Parse(parts[i], CultureInfo.InvariantCulture);
        }
        return result;
    }
    
    static void Main()
    {
        // Use buffered I/O for speed
        reader = new StreamReader(Console.OpenStandardInput(), Encoding.ASCII, false, 1 << 16);
        output = new StringBuilder();
        
        // Parse first line
        double[] firstLine = ReadDoubles();
        n = (int)firstLine[0];
        epsilon = firstLine[1];
        
        x = new double[n];
        y = new double[n];
        
        for (int i = 0; i < n; i++)
        {
            double[] coords = ReadDoubles();
            x[i] = coords[0];
            y[i] = coords[1];
        }
        
        // Special case: single point
        if (n == 1)
        {
            Console.WriteLine("1");
            Console.WriteLine("1");
            return;
        }
        
        // Special case: e = 2 (maximum Manhattan distance in unit square is 2)
        // Any single point covers the entire square within e
        if (epsilon >= 2.0)
        {
            Console.WriteLine("1");
            Console.WriteLine("1");
            return;
        }
        
        // Compute MST
        ComputeMST();
        
        // Sample interval: slightly under e/2 for numerical safety
        // Using 0.49e ensures we don't miss coverage due to floating-point errors
        double sampleThreshold = epsilon * 0.49;
        
        // Maximum allowed points
        double maxPointsDouble = 18.0 / (epsilon * epsilon);
        int maxPoints = (int)Math.Ceiling(maxPointsDouble);
        
        // Root MST at vertex 0
        int[] parent = new int[n];
        List<int>[] children = new List<int>[n];
        
        for (int i = 0; i < n; i++)
        {
            parent[i] = -1;
            children[i] = new List<int>();
        }
        
        // BFS to build parent-child structure
        bool[] visited = new bool[n];
        Queue<int> bfs = new Queue<int>();
        bfs.Enqueue(0);
        visited[0] = true;
        
        while (bfs.Count > 0)
        {
            int u = bfs.Dequeue();
            foreach (int v in mstAdj[u])
            {
                if (!visited[v])
                {
                    visited[v] = true;
                    parent[v] = u;
                    children[u].Add(v);
                    bfs.Enqueue(v);
                }
            }
        }
        
        // DFS to select vertices for Q
        List<int> Q = new List<int>();
        List<(int from, int to)> T = new List<(int, int)>();
        
        int[] lastSelectedAncestor = new int[n];
        double[] distFromLastSelected = new double[n];
        
        // Select root (always included)
        Q.Add(0);
        lastSelectedAncestor[0] = 0;
        distFromLastSelected[0] = 0;
        
        // Iterative DFS
        Stack<int> stack = new Stack<int>();
        int[] nextChild = new int[n];
        stack.Push(0);
        
        while (stack.Count > 0)
        {
            int u = stack.Peek();
            
            if (nextChild[u] < children[u].Count)
            {
                int v = children[u][nextChild[u]];
                nextChild[u]++;
                
                double edgeDist = ManhattanDist(u, v);
                double cumDist = distFromLastSelected[u] + edgeDist;
                
                // Selection criteria:
                // 1. Leaves must be selected (endpoints of MST)
                // 2. Branching points must be selected (to maintain tree structure)
                // 3. Distance threshold: when cumulative distance = sampleThreshold
                // 4. Long edges: if single edge exceeds threshold
                bool isLeaf = children[v].Count == 0;
                bool isBranching = children[v].Count >= 2;
                bool distThreshold = cumDist >= sampleThreshold;
                bool longEdge = edgeDist >= sampleThreshold;
                
                if (isLeaf || isBranching || distThreshold || longEdge)
                {
                    Q.Add(v);
                    T.Add((lastSelectedAncestor[u], v));
                    lastSelectedAncestor[v] = v;
                    distFromLastSelected[v] = 0;
                }
                else
                {
                    // Inherit from parent
                    lastSelectedAncestor[v] = lastSelectedAncestor[u];
                    distFromLastSelected[v] = cumDist;
                }
                
                stack.Push(v);
            }
            else
            {
                stack.Pop();
            }
        }
        
        // Verify constraint (should always pass, but good for debugging)
        if (Q.Count > maxPoints)
        {
            // This shouldn't happen with proper algorithm, but just in case
            // Fall back to taking all MST edges (which might exceed limit)
            Console.Error.WriteLine($"Warning: Q.Count={Q.Count} exceeds maxPoints={maxPoints}");
        }
        
        // Build position index for Q (1-based indexing within Q)
        Dictionary<int, int> qPos = new Dictionary<int, int>();
        for (int i = 0; i < Q.Count; i++)
        {
            qPos[Q[i]] = i + 1;
        }
        
        // Output
        output.AppendLine(Q.Count.ToString());
        
        // Output Q vertices (1-based original vertex indices)
        for (int i = 0; i < Q.Count; i++)
        {
            if (i > 0) output.Append(' ');
            output.Append(Q[i] + 1); // Convert to 1-based
        }
        output.AppendLine();
        
        // Output tree edges (1-based position indices within Q)
        foreach (var (from, to) in T)
        {
            output.Append(qPos[from]);
            output.Append(' ');
            output.AppendLine(qPos[to].ToString());
        }
        
        Console.Write(output.ToString());
    }
}